<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Patient;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class PatientRegistrationController extends Controller
{
    public function showRegistration()
    {
        // Check if patient registration is enabled
        if (!Setting::get('patient_registration_enabled', false)) {
            abort(403, 'Patient registration is currently disabled.');
        }

        return view('patient.register');
    }

    public function register(Request $request)
    {
        // Check if patient registration is enabled
        if (!Setting::get('patient_registration_enabled', false)) {
            abort(403, 'Patient registration is currently disabled.');
        }

        $validated = $request->validate([
            'title' => 'nullable|string',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'date_of_birth' => 'required|date',
            'gender' => 'required|in:Male,Female,Other',
            'identity_number' => 'nullable|string|unique:patients,identity_number',
            'phone_area_code' => 'nullable|string',
            'phone_number' => 'required|string',
            'email' => 'required|email|unique:users,email|unique:patients,email',
            'password' => 'required|string|min:8|confirmed',
            'payment_mode' => 'required|in:Cash Payment,Medical Aid Payment',
            'medical_aid_number' => 'nullable|string|required_if:payment_mode,Medical Aid Payment',
            'main_member' => 'nullable|string',
        ]);

        // Create user account
        $user = User::create([
            'name' => "{$validated['first_name']} {$validated['last_name']}",
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'patient',
            'is_active' => true,
        ]);

        // Create patient record
        $patient = Patient::create([
            'user_id' => $user->id,
            'patient_id' => 'PAT' . str_pad(Patient::count() + 1, 8, '0', STR_PAD_LEFT),
            'title' => $validated['title'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'date_of_birth' => $validated['date_of_birth'],
            'gender' => $validated['gender'],
            'identity_number' => $validated['identity_number'] ?? Str::random(10),
            'phone_area_code' => $validated['phone_area_code'],
            'phone_number' => $validated['phone_number'],
            'email' => $validated['email'],
            'payment_mode' => $validated['payment_mode'],
            'medical_aid_number' => $validated['medical_aid_number'] ?? null,
            'main_member' => $validated['main_member'] ?? null,
        ]);

        return redirect()->route('login')
            ->with('success', 'Registration successful! You can now login to view your prescriptions and appointments.');
    }
}

